/* 
 * rw.c 
 *
 * Копіює задану кількість байтів із одного файлу в другий.
 * Ілюструє порядок застосування функцій open(), read(), write(), close().
 *
 */

#include <errno.h>
#include <fcntl.h>
#include <libgen.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>


/* Читає count байтів із файлу filename. 
   Повертає покажчик на буфер із прочитаними даними. */
char *read_from_file(const char *filename, size_t count)
{
        char *buf;
        int fd;

        /* Розміщує буфер */
        buf = malloc(count);
        if (buf == NULL) {
                /* Розмістити буфер не вдалось.
                   Виводить повідомлення про помилку */
                fprintf(stderr, "Not enough memory\n");
                /* і повертає управління. */
                return NULL;
        }

        /* Відкриває файл для читання. */
        fd = open(filename, O_RDONLY);
        if (fd < 0) {
                /* Файл відкрити не вдалось.
                   Виводить повідомлення про помилку, */
                perror("open()");
                /* звільняє буфер */
                free(buf);
                /* і повертає управління. */
                return NULL;
        }

        /* Читає дані. */
        if (read(fd, buf, count) < 0) {
                /* Помилка читання.
                   Виводить повідомлення про помилку, */
                perror("read()");
                /* Звільняє буфер, */
                free(buf);
                /* закриває файл */
                if (close(fd) != 0) 
                        perror("close()");
                /* і повертає управління. */
                return NULL;
        }

        /* Закриває файл. */
        if (close(fd) != 0) 
                perror("close()");
        /* Повертає управління. */
        return buf;
}


/* Записує count байтів із буфера buf у файл filename. */
size_t write_to_file(const char *filename, char *buf, size_t count)
{
        int fd;
        size_t rcount;

        /* Відкриває файл для запису. */
        fd = open(filename, O_WRONLY | O_CREAT | O_EXCL, 00666);
        if (fd < 0) {
                /* Відкрити файл не вдалось. */
                /* Виводить повідомлення про помилку */
                perror("open()");
                /* і повертає управління. */
                return 0;
        }

        /* Записує дані. */
        rcount = write(fd, buf, count);
        if (rcount < 0) {
                /* Помилка запису. */
                /* Виводить повідомлення про помилку, */
                perror("write()");
                /* закриває файл */
                if (close(fd) != 0) 
                        perror("close()");
                /* і повертає управління. */
                return 0;
        }

        /* Закриває файл. */
        if (close(fd) != 0) 
                perror("close()");
        /* Повертає управління. */
        return rcount;
}

int main(int argc, char *argv[])
{
        char *src = argv[1];
        char *dst = argv[2];
        size_t count;
        char *buf;

        if (argc < 4) {
                /* Виводить повідомлення про порядок запуску. */
                fprintf(stderr, "Usage: %s source dest count\n",
                                                basename(argv[0]));
                exit(EXIT_FAILURE);
        }

        /* Перетворює розмір у числову форму. */
        count = (size_t) atoi(argv[3]);

        /* Читає дані з початкового файлу. */
        buf = read_from_file(src, count);
        if (buf == NULL) {
                /* Повідомляє про помилку читання і завершує роботу. */
                fprintf(stderr, "Can't read %d bytes from %s\n",
                                                        count, src);
                exit(EXIT_FAILURE);
        }
        /* Повідомляє про успіх читання. */
        printf("%d bytes from %s has been read\n", count, src);

        /* Записує дані у результуючий файл. */
        if (write_to_file(dst, buf, count) != count) {
                /* Повідомляє про помилку запису і завершує роботу. */
                fprintf(stderr, "Can't write %d bytes to %s\n",
                                                        count, dst);
                exit(EXIT_FAILURE);
        }
        /* Повідомляє про успіх запису. */
        printf("%d bytes to %s has been written\n", count, dst);

        /* Звільняє буфер. */
        /* Потрібно робити тільки тоді, коли ще повинна виконуватись 
           якась робота (а не відразу exit()). */
        free(buf);                  
        /* Завершує роботу. */
        exit(EXIT_SUCCESS);
}
